<?php
// serviceability.php
// Single-file UI + backend for Shiprocket Hyperlocal Serviceability check
// NOTE: Credentials shown are taken from user's snippet. Replace if needed.

class ShiprocketAPI {
    private $user_name;
    private $password;

    public function __construct() {
        // Use provided credentials (replace for security in production)
        $this->user_name = "drodifytechnology@gmail.com";
        $this->password = "!Ml6nwu1bBr^FVEm";
    }

    // Get auth token
    public function getToken() {
        $user = [
            "email" => $this->user_name,
            "password" => $this->password,
        ];
        $login_user = json_encode($user);

        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://apiv2.shiprocket.in/v1/external/auth/login",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => $login_user,
            CURLOPT_HTTPHEADER => array(
                "Content-Type: application/json"
            ),
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err) {
            return ["status" => false, "message" => "Curl error: $err"];
        }

        $res = json_decode($response);
        if (isset($res->token)) {
            return ["status" => true, "token" => $res->token];
        } else {
            return ["status" => false, "message" => $res->message ?? ($response ?: "Unknown error")];
        }
    }

    // Check serviceability (hyperlocal)
  public function checkServiceability($token, $payload) {
    // Build query string instead of POST
    $query = http_build_query($payload);

    $curl = curl_init();
    curl_setopt_array($curl, array(
        CURLOPT_URL => "https://apiv2.shiprocket.in/v1/external/courier/serviceability/?" . $query,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
        CURLOPT_HTTPHEADER => array(
            "Authorization: Bearer " . $token,
        ),
    ));

    $response = curl_exec($curl);
    $err = curl_error($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);

    if ($err) {
        return ["status" => false, "message" => "Curl error: $err"];
    }
    $decoded = json_decode($response, true);
    return ["status" => true, "http_code" => $http_code, "raw" => $response, "json" => $decoded];
  }

}

// Handle POST
$errors = [];
$result = null;
$submitted_payload = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Collect and sanitize inputs
    $pickup_postcode = isset($_POST['pickup_postcode']) ? trim($_POST['pickup_postcode']) : '';
    $delivery_postcode = isset($_POST['delivery_postcode']) ? trim($_POST['delivery_postcode']) : '';
    $order_id = isset($_POST['order_id']) && $_POST['order_id'] !== '' ? (int)$_POST['order_id'] : null;
    $cod = isset($_POST['cod']) ? 1 : 0;
    $weight = isset($_POST['weight']) && $_POST['weight'] !== '' ? (float)$_POST['weight'] : null;
    $is_new_hyperlocal = isset($_POST['is_new_hyperlocal']) ? 1 : 0;

    $lat_from = isset($_POST['lat_from']) && $_POST['lat_from'] !== '' ? (float)$_POST['lat_from'] : null;
    $long_from = isset($_POST['long_from']) && $_POST['long_from'] !== '' ? (float)$_POST['long_from'] : null;
    $lat_to = isset($_POST['lat_to']) && $_POST['lat_to'] !== '' ? (float)$_POST['lat_to'] : null;
    $long_to = isset($_POST['long_to']) && $_POST['long_to'] !== '' ? (float)$_POST['long_to'] : null;

    // Basic validation
    if ($pickup_postcode === '') $errors[] = "Pickup postcode is required.";
    if ($delivery_postcode === '') $errors[] = "Delivery postcode is required.";
    // Either order_id OR (cod + weight) required (per note)
    if (!$order_id && ($weight === null || $weight <= 0)) {
        $errors[] = "Either provide an order_id or a valid weight (and cod if COD).";
    }
    if ($is_new_hyperlocal) {
        if ($lat_from === null || $long_from === null || $lat_to === null || $long_to === null) {
            $errors[] = "For hyperlocal shipments you must provide lat/long for pickup and delivery.";
        }
    }

    if (empty($errors)) {
        // build payload following API fields
        $payload = [
            "pickup_postcode" => (int)$pickup_postcode,
            "delivery_postcode" => (int)$delivery_postcode,
            "is_new_hyperlocal" => (int)$is_new_hyperlocal,
        ];
        if ($order_id) $payload["order_id"] = (int)$order_id;
        if ($weight !== null && $weight > 0) $payload["weight"] = $weight;
        if (isset($_POST['cod'])) $payload["cod"] = (int)$cod;
        if ($is_new_hyperlocal) {
            $payload["lat_from"] = $lat_from;
            $payload["long_from"] = $long_from;
            $payload["lat_to"] = $lat_to;
            $payload["long_to"] = $long_to;
        }

        $submitted_payload = $payload;

        // call Shiprocket
        $api = new ShiprocketAPI();
        $token_res = $api->getToken();
        if (!$token_res['status']) {
            $errors[] = "Auth failed: " . ($token_res['message'] ?? 'Unknown');
        } else {
            $token = $token_res['token'];
            $res = $api->checkServiceability($token, $payload);
            $result = $res;
        }
    }
}

?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Hyperlocal — Check Courier Serviceability</title>
<link rel="preconnect" href="https://fonts.gstatic.com">
<style>
    :root{
        --bg:#0f1724;
        --card:#0b1220;
        --muted:#94a3b8;
        --accent:#06b6d4;
        --glass: rgba(255,255,255,0.03);
        --success:#10b981;
        --danger:#ef4444;
    }
    *{box-sizing:border-box;font-family:Inter,ui-sans-serif,system-ui,-apple-system,'Segoe UI',Roboto,'Helvetica Neue',Arial;}
    body{margin:0;min-height:100vh;background:
        radial-gradient(1200px 400px at 10% 10%, rgba(6,182,212,0.06), transparent 10%),
        radial-gradient(900px 300px at 90% 90%, rgba(16,185,129,0.04), transparent 10%),
        var(--bg);
        color:#e6eef6;
        padding:32px;
        -webkit-font-smoothing:antialiased;
    }
    .container{max-width:1100px;margin:0 auto;display:grid;grid-template-columns:1fr 420px;gap:28px;}
    header{grid-column:1/-1;margin-bottom:8px;display:flex;align-items:center;justify-content:space-between}
    header h1{margin:0;font-size:20px;letter-spacing:0.2px}
    header p{margin:0;color:var(--muted);font-size:13px}

    .card{background:linear-gradient(180deg, rgba(255,255,255,0.02), rgba(255,255,255,0.01));padding:20px;border-radius:12px;box-shadow:0 6px 18px rgba(2,6,23,0.6);border:1px solid rgba(255,255,255,0.03);}
    form .row{display:flex;gap:12px;margin-bottom:12px}
    label{font-size:13px;color:var(--muted);display:block;margin-bottom:6px}
    input[type="text"], input[type="number"], input[type="float"] , input[type="email"]{
        width:100%;padding:10px;border-radius:8px;border:1px solid rgba(255,255,255,0.04);background:var(--glass);color:inherit;
    }
    .small{font-size:12px;color:var(--muted);margin-top:6px}
    .col{flex:1;min-width:0}
    .actions{display:flex;gap:10px;align-items:center;margin-top:6px}
    .btn{background:linear-gradient(90deg,var(--accent),#3b82f6);padding:10px 14px;border-radius:10px;color:#021124;border:none;cursor:pointer;font-weight:600}
    .btn-ghost{background:transparent;border:1px solid rgba(255,255,255,0.04);padding:10px 12px;border-radius:10px;color:var(--muted);cursor:pointer}
    .muted{color:var(--muted)}
    .switch {display:inline-flex;align-items:center;gap:8px}
    .switch input{width:18px;height:18px}
    .note{font-size:13px;color:var(--muted);margin-bottom:10px}

    /* Response pane */
    .response {overflow:auto;max-height:620px;padding:16px;border-radius:10px;background:linear-gradient(180deg, rgba(0,0,0,0.12), transparent);}
    pre{white-space:pre-wrap;word-wrap:break-word;font-size:13px;background:transparent;border:0;padding:0}
    .error{background:rgba(239,68,68,0.08);border-left:4px solid var(--danger);padding:10px;border-radius:6px;color:var(--danger)}
    .success{background:rgba(16,185,129,0.06);border-left:4px solid var(--success);padding:10px;border-radius:6px;color:var(--success)}
    .result-table{width:100%;border-collapse:collapse;margin-top:12px}
    .result-table th, .result-table td{padding:8px 10px;text-align:left;border-bottom:1px dashed rgba(255,255,255,0.03);font-size:13px}
    .hint{font-size:12px;color:var(--muted);margin-top:8px}
    @media (max-width:980px){
        .container{grid-template-columns:1fr; padding:16px}
    }
</style>
</head>
<body>
<div class="container">
    <header>
        <div>
            <h1>Hyperlocal — Check Courier Serviceability</h1>
            <p>Enter pickup / delivery postcodes and hyperlocal coordinates. Uses Shiprocket (cURL)</p>
        </div>
        <div class="muted">Demo UI · cURL · PHP</div>
    </header>

    <main class="card">
        <form method="post" id="serviceabilityForm">
            <div class="note">Tip: Provide either <strong>Order ID</strong> OR <strong>Weight</strong> (and COD if needed). For hyperlocal, toggle the checkbox and provide lat/longs.</div>

            <div class="row">
                <div class="col">
                    <label for="pickup_postcode">Pickup Postcode</label>
                    <input id="pickup_postcode" name="pickup_postcode" type="text" placeholder="110030" required value="<?= htmlspecialchars($_POST['pickup_postcode'] ?? '') ?>">
                </div>
                <div class="col">
                    <label for="delivery_postcode">Delivery Postcode</label>
                    <input id="delivery_postcode" name="delivery_postcode" type="text" placeholder="122002" required value="<?= htmlspecialchars($_POST['delivery_postcode'] ?? '') ?>">
                </div>
            </div>

            <div class="row">
                <div class="col">
                    <label for="order_id">Order ID (optional)</label>
                    <input id="order_id" name="order_id" type="number" placeholder="123456" value="<?= htmlspecialchars($_POST['order_id'] ?? '') ?>">
                    <div class="small">If you already have a Shiprocket order id, supply it. Otherwise leave blank and use weight + cod.</div>
                </div>
                <div class="col">
                    <label for="weight">Weight (kg)</label>
                    <input id="weight" name="weight" type="text" placeholder="0.5" value="<?= htmlspecialchars($_POST['weight'] ?? '') ?>">
                    <div class="small">Numeric. Required if order_id not provided.</div>
                </div>
            </div>

            <div class="row" style="align-items:center;">
                <label class="switch">
                    <input type="checkbox" id="cod" name="cod" <?= isset($_POST['cod']) ? 'checked' : '' ?>>
                    <span class="small">Cash on Delivery (COD)</span>
                </label>

                <label class="switch" style="margin-left:16px;">
                    <input type="checkbox" id="is_new_hyperlocal" name="is_new_hyperlocal" <?= isset($_POST['is_new_hyperlocal']) ? 'checked' : '' ?> onchange="toggleHyperlocal()">
                    <span class="small">Is New Hyperlocal</span>
                </label>
            </div>

            <div id="hyperlocalFields" style="display: <?= isset($_POST['is_new_hyperlocal']) ? 'block' : 'none' ?>; margin-top:12px;">
                <div class="row">
                    <div class="col">
                        <label for="lat_from">Pickup Latitude</label>
                        <input id="lat_from" name="lat_from" type="text" placeholder="28.509223937988" value="<?= htmlspecialchars($_POST['lat_from'] ?? '') ?>">
                    </div>
                    <div class="col">
                        <label for="long_from">Pickup Longitude</label>
                        <input id="long_from" name="long_from" type="text" placeholder="77.067848205566" value="<?= htmlspecialchars($_POST['long_from'] ?? '') ?>">
                    </div>
                </div>
                <div class="row">
                    <div class="col">
                        <label for="lat_to">Delivery Latitude</label>
                        <input id="lat_to" name="lat_to" type="text" placeholder="28.50724220275879" value="<?= htmlspecialchars($_POST['lat_to'] ?? '') ?>">
                    </div>
                    <div class="col">
                        <label for="long_to">Delivery Longitude</label>
                        <input id="long_to" name="long_to" type="text" placeholder="77.06745147705078" value="<?= htmlspecialchars($_POST['long_to'] ?? '') ?>">
                    </div>
                </div>
            </div>

            <div class="actions">
                <button type="submit" class="btn">Check Serviceability</button>
                <button type="button" class="btn-ghost" onclick="resetForm()">Reset</button>
                <div style="flex:1"></div>
                <div class="hint">Response will appear on the right panel.</div>
            </div>
        </form>
    </main>

    <aside class="card">
        <h3 style="margin-top:0">Response</h3>

        <div class="response">
            <?php if (!empty($errors)): ?>
                <div class="error">
                    <strong>Validation / Request errors:</strong>
                    <ul style="margin:8px 0 0 18px;padding:0">
                        <?php foreach ($errors as $e): ?>
                            <li><?= htmlspecialchars($e) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php elseif ($result === null): ?>
                <div class="muted">No request yet. Fill the form and press <strong>Check Serviceability</strong>.</div>
            <?php else: ?>
                <?php if (!$result['status']): ?>
                    <div class="error">
                        <strong>Error:</strong> <?= htmlspecialchars($result['message'] ?? 'Unknown error') ?>
                    </div>
                <?php else: ?>
                    <div class="success">
                        <strong>HTTP Status:</strong> <?= htmlspecialchars($result['http_code'] ?? '200') ?>
                    </div>

                    <?php if (!empty($submitted_payload)): ?>
                        <div style="margin-top:12px">
                            <div class="muted">Payload sent:</div>
                            <pre><?= htmlspecialchars(json_encode($submitted_payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)) ?></pre>
                        </div>
                    <?php endif; ?>

                    <div style="margin-top:12px">
                        <div class="muted">Raw response:</div>
                        <pre><?= htmlspecialchars($result['raw']) ?></pre>
                    </div>

                    <?php
                    // If json-couriers exist show table (common response shape = list of couriers)
                    $data = $result['json'] ?? null;
                    ?>
          <?php if (is_array($data) && !empty($data['data']) && is_array($data['data'])): ?>
    <table class="result-table" style="margin-top:12px">
        <thead>
            <tr>
                <th>Courier</th>
                <th>Estimated Delivery</th>
                <th>Rate</th>
                <th>RTO Rate</th>
                <th>ETD Hours</th>
                <th>Distance (km)</th>
            </tr>
        </thead>
        <tbody>
        <?php foreach ($data['data'] as $item): ?>
            <tr>
                <td><?= htmlspecialchars($item['courier_name'] ?? '—') ?></td>
                <td><?= htmlspecialchars($item['etd'] ?? '—') ?></td>
                <td><?= htmlspecialchars($item['rates'] ?? '—') ?></td>
                <td><?= htmlspecialchars($item['rto_rates'] ?? '—') ?></td>
                <td><?= htmlspecialchars($item['etd_hours'] ?? '—') ?></td>
                <td><?= htmlspecialchars($item['distance'] ?? '—') ?></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>


                <?php endif; ?>
            <?php endif; ?>
        </div>
    </aside>

</div>

<script>
    function toggleHyperlocal() {
        const cb = document.getElementById('is_new_hyperlocal');
        const panel = document.getElementById('hyperlocalFields');
        panel.style.display = cb.checked ? 'block' : 'none';
    }

    function resetForm() {
        if (!confirm('Reset all fields?')) return;
        document.getElementById('serviceabilityForm').reset();
        toggleHyperlocal();
    }

    // Simple client-side pre-check to avoid submitting obviously incomplete forms
    document.getElementById('serviceabilityForm').addEventListener('submit', function (e) {
        const pickup = document.getElementById('pickup_postcode').value.trim();
        const delivery = document.getElementById('delivery_postcode').value.trim();
        const orderId = document.getElementById('order_id').value.trim();
        const weight = document.getElementById('weight').value.trim();
        const isHyper = document.getElementById('is_new_hyperlocal').checked;

        let messages = [];
        if (!pickup) messages.push('Pickup postcode is required.');
        if (!delivery) messages.push('Delivery postcode is required.');

        if (!orderId && (!weight || parseFloat(weight) <= 0)) {
            messages.push('Either order id OR a valid weight is required.');
        }

        if (isHyper) {
            const latFrom = document.getElementById('lat_from').value.trim();
            const longFrom = document.getElementById('long_from').value.trim();
            const latTo = document.getElementById('lat_to').value.trim();
            const longTo = document.getElementById('long_to').value.trim();
            if (!latFrom || !longFrom || !latTo || !longTo) {
                messages.push('For hyperlocal shipments provide all lat/long values.');
            }
        }

        if (messages.length) {
            e.preventDefault();
            alert('Fix these issues before submitting:\n\n' + messages.join('\n'));
        }
    });
</script>
</body>
</html>
